#!/usr/bin/env bash
# Ubuntu CIS(-ish) Hardening Script — Level 1 baseline + opt-in Level 2
# Tested on Ubuntu 20.04/22.04/24.04 (server). Run as root.
# LOG: /var/log/harden.log

set -euo pipefail

# ======= TOGGLES (adjust to taste) ===========================================
DISABLE_IPV6="no"                   # "yes" to disable IPv6 (CIS L2-ish)
SSH_PORT="22"                       # change if you use a non-standard port
ALLOW_ADDITIONAL_SSH_FROM_CIDR=""   # e.g. "10.0.0.0/8 192.168.0.0/16"
USE_UFW="yes"
ENABLE_FAIL2BAN="yes"
ENABLE_AIDE="yes"                   # file integrity monitoring
ENABLE_AUDITD_RULESET="yes"
HARDEN_SSH_STRICT="yes"             # ciphers/MACs/KEX, root/pw auth disabled
AUTO_UPGRADES_REBOOT="yes"          # reboot if needed after unattended-upgrades
PURGE_LEGACY_SERVICES="yes"         # telnet, rsh, tftp, nis, xinetd, etc.
PURGE_AVAHI_CUPS_DESKTOPY="no"      # set "yes" on servers, "no" on desktops
SEPARATE_TMP_TMPFS="no"             # safer with proper change mgmt; opt-in
RESTRICT_AT_CRON="yes"
STRONG_PW_POLICY="yes"
PW_MAX_DAYS="365"                   # CIS often <=365 (L1), <=90 (stricter)
PW_MIN_DAYS="1"
PW_WARN_DAYS="7"
LOCKOUT_AFTER_FAILS="yes"           # PAM faillock
LOCKOUT_TRIES="5"
LOCKOUT_SECONDS="900"               # 15 minutes
UMASK_DEFAULT="027"                 # tighter than Ubuntu default 022
DISABLE_USB_STORAGE="no"            # set to "yes" to blacklist usb-storage
NTP_IMPL="chrony"                   # "chrony" (recommended) or "systemd-timesyncd"
BANNER_TEXT="Authorized use only. Activity may be monitored and reported."

# ======= Helpers ==============================================================
log(){ echo "[$(date +'%F %T')] $*" | tee -a /var/log/harden.log; }
edit(){ local f="$1"; [ -f "$f" ] && cp -a "$f" "${f}.bak.$(date +%F-%H%M%S)"; }
ensure_line(){ local f="$1"; shift; local line="$*"; grep -qsF -- "$line" "$f" || echo "$line" >> "$f"; }
pkg_installed(){ dpkg -s "$1" &>/dev/null; }

[ "$(id -u)" -eq 0 ] || { echo "Run as root."; exit 1; }
mkdir -p /var/log

log "Starting Ubuntu hardening…"
log "Detected: $(lsb_release -ds || true)"

# ======= 0) Update & base pkgs ===============================================
log "Updating system packages…"
export DEBIAN_FRONTEND=noninteractive
apt-get update -y
apt-get dist-upgrade -y
apt-get install -y curl wget gnupg ca-certificates apt-transport-https

# ======= 1) Time sync ========================================================
if [[ "$NTP_IMPL" == "chrony" ]]; then
  log "Installing chrony…"
  apt-get install -y chrony
  systemctl enable --now chrony
else
  log "Configuring systemd-timesyncd…"
  apt-get install -y systemd-timesyncd || true
  systemctl enable --now systemd-timesyncd
fi

# ======= 2) Unattended security updates =====================================
log "Enabling unattended-upgrades…"
apt-get install -y unattended-upgrades update-notifier-common
dpkg-reconfigure -f noninteractive unattended-upgrades
mkdir -p /etc/apt/apt.conf.d
cat >/etc/apt/apt.conf.d/51-auto-reboot.conf <<EOF
Unattended-Upgrade::Automatic-Reboot "$( [[ "$AUTO_UPGRADES_REBOOT" == "yes" ]] && echo "true" || echo "false" )";
Unattended-Upgrade::Automatic-Reboot-Time "03:30";
EOF

# ======= 3) Firewall (UFW) ===================================================
if [[ "$USE_UFW" == "yes" ]]; then
  log "Configuring UFW…"
  apt-get install -y ufw
  ufw --force reset
  ufw default deny incoming
  ufw default allow outgoing
  ufw allow "${SSH_PORT}/tcp"
  for cidr in $ALLOW_ADDITIONAL_SSH_FROM_CIDR; do ufw allow from "$cidr" to any port "$SSH_PORT" proto tcp; done
  ufw --force enable
fi

# ======= 4) SSH hardening ====================================================
log "Hardening SSH…"
apt-get install -y openssh-server
SSHD="/etc/ssh/sshd_config"
edit "$SSHD"

# baseline
ensure_line "$SSHD" "Port ${SSH_PORT}"
ensure_line "$SSHD" "Protocol 2"
ensure_line "$SSHD" "LoginGraceTime 60"
ensure_line "$SSHD" "MaxAuthTries 3"
ensure_line "$SSHD" "ClientAliveInterval 300"
ensure_line "$SSHD" "ClientAliveCountMax 0"
ensure_line "$SSHD" "PermitRootLogin no"
ensure_line "$SSHD" "X11Forwarding no"
ensure_line "$SSHD" "AllowTcpForwarding no"
ensure_line "$SSHD" "PermitEmptyPasswords no"
ensure_line "$SSHD" "UsePAM yes"
ensure_line "$SSHD" "PasswordAuthentication no"
ensure_line "$SSHD" "KbdInteractiveAuthentication no"
ensure_line "$SSHD" "ChallengeResponseAuthentication no"
ensure_line "$SSHD" "PubkeyAuthentication yes"
ensure_line "$SSHD" "AllowAgentForwarding no"
ensure_line "$SSHD" "PrintLastLog yes"
ensure_line "$SSHD" "LogLevel VERBOSE"
ensure_line "$SSHD" "Banner /etc/issue.net"

if [[ "$HARDEN_SSH_STRICT" == "yes" ]]; then
  ensure_line "$SSHD" "Ciphers chacha20-poly1305@openssh.com,aes256-gcm@openssh.com,aes256-ctr"
  ensure_line "$SSHD" "MACs hmac-sha2-512-etm@openssh.com,hmac-sha2-256-etm@openssh.com"
  ensure_line "$SSHD" "KexAlgorithms curve25519-sha256,curve25519-sha256@libssh.org"
  ensure_line "$SSHD" "HostKeyAlgorithms ssh-ed25519,ssh-ed25519-cert-v01@openssh.com"
fi

systemctl reload ssh || systemctl restart ssh

# ======= 5) Fail2ban =========================================================
if [[ "$ENABLE_FAIL2BAN" == "yes" ]]; then
  log "Installing Fail2ban…"
  apt-get install -y fail2ban
  mkdir -p /etc/fail2ban
  edit /etc/fail2ban/jail.local
  cat >/etc/fail2ban/jail.local <<EOF
[DEFAULT]
bantime = 1h
findtime = 10m
maxretry = 5
backend = systemd
[sshd]
enabled = true
port = ${SSH_PORT}
mode = aggressive
EOF
  systemctl enable --now fail2ban
fi

# ======= 6) Password & account policies =====================================
if [[ "$STRONG_PW_POLICY" == "yes" ]]; then
  log "Configuring password policy…"
  apt-get install -y libpam-pwquality
  edit /etc/security/pwquality.conf
  cat >/etc/security/pwquality.conf <<'EOF'
minlen = 14
dcredit = -1
ucredit = -1
lcredit = -1
ocredit = -1
minclass = 3
maxrepeat = 3
reject_username = 1
enforce_for_root
EOF

  # login.defs aging
  edit /etc/login.defs
  sed -ri "s/^\s*PASS_MAX_DAYS\s+.*/PASS_MAX_DAYS\t$PW_MAX_DAYS/" /etc/login.defs || true
  sed -ri "s/^\s*PASS_MIN_DAYS\s+.*/PASS_MIN_DAYS\t$PW_MIN_DAYS/" /etc/login.defs || true
  sed -ri "s/^\s*PASS_WARN_AGE\s+.*/PASS_WARN_AGE\t$PW_WARN_DAYS/" /etc/login.defs || true

  # PAM faillock (18.04+ style; Ubuntu uses pam_faillock in common-auth/account)
  for f in /etc/pam.d/common-auth /etc/pam.d/common-account; do
    edit "$f"
  done
  awk -v t="$LOCKOUT_TRIES" -v s="$LOCKOUT_SECONDS" '
    BEGIN{ print "auth required pam_faillock.so preauth silent deny=" t " unlock_time=" s }
    { print }
    END{ print "auth [default=die] pam_faillock.so authfail deny=" t " unlock_time=" s
         print "account required pam_faillock.so" }' /etc/pam.d/common-auth > /etc/pam.d/common-auth.new
  mv /etc/pam.d/common-auth.new /etc/pam.d/common-auth
fi

# require strong umask
log "Setting system umask to $UMASK_DEFAULT…"
edit /etc/profile
ensure_line /etc/profile "umask $UMASK_DEFAULT"
edit /etc/login.defs
sed -ri "s/^\s*UMASK\s+.*/UMASK\t$UMASK_DEFAULT/" /etc/login.defs || ensure_line /etc/login.defs "UMASK $UMASK_DEFAULT"

# ======= 7) Disable core dumps & limits =====================================
log "Disabling core dumps and hardening limits…"
edit /etc/security/limits.conf
ensure_line /etc/security/limits.conf "* hard core 0"
ensure_line /etc/security/limits.conf "* soft core 0"
edit /etc/systemd/coredump.conf
ensure_line /etc/systemd/coredump.conf "[Coredump]"
ensure_line /etc/systemd/coredump.conf "Storage=none"
ensure_line /etc/systemd/coredump.conf "ProcessSizeMax=0"
systemctl restart systemd-coredump.service || true

# ======= 8) Auditd ===========================================================
log "Installing and configuring auditd…"
apt-get install -y auditd audispd-plugins
systemctl enable --now auditd

if [[ "$ENABLE_AUDITD_RULESET" == "yes" ]]; then
  edit /etc/audit/rules.d/hardening.rules
  cat >/etc/audit/rules.d/hardening.rules <<'EOF'
# CIS-inspired audit rules (common subset)
-D
-b 8192
# Time changes
-a always,exit -F arch=b64 -S adjtimex,settimeofday,clock_settime -k time-change
-w /etc/localtime -p wa -k time-change
# Identity changes
-w /etc/group -p wa -k identity
-w /etc/passwd -p wa -k identity
-w /etc/shadow -p wa -k identity
# Network changes
-w /etc/hosts -p wa -k network
-w /etc/hostname -p wa -k network
-w /etc/resolv.conf -p wa -k network
# Priv esc
-a always,exit -F arch=b64 -S setuid,setgid -k priv-change
-w /etc/sudoers -p wa -k scope
-w /etc/sudoers.d/ -p wa -k scope
# Login events
-w /var/log/faillog -p wa -k logins
-w /var/log/lastlog -p wa -k logins
# Kernel mods
-w /sbin/insmod -p x -k modules
-w /sbin/rmmod -p x -k modules
-w /sbin/modprobe -p x -k modules
# MAC policy
-w /etc/apparmor/ -p wa -k MAC-policy
# System admin activity
-w /var/log/sudo -p wa -k actions
# Immutable at end
-e 2
EOF
  augenrules --load
  systemctl restart auditd
fi

# ======= 9) AppArmor =========================================================
log "Ensuring AppArmor is enabled…"
apt-get install -y apparmor apparmor-utils
systemctl enable --now apparmor || true

# ======= 10) AIDE ============================================================
if [[ "$ENABLE_AIDE" == "yes" ]]; then
  log "Installing AIDE…"
  apt-get install -y aide
  aideinit || aide --init || true
  if [ -f /var/lib/aide/aide.db.new ]; then
    mv /var/lib/aide/aide.db.new /var/lib/aide/aide.db
  fi
  (crontab -l 2>/dev/null; echo "0 4 * * * /usr/bin/aide.wrapper --check") | crontab -
fi

# ======= 11) Sysctl network stack ===========================================
log "Applying kernel network hardening…"
SYSCTL="/etc/sysctl.d/99-hardening.conf"
edit "$SYSCTL"
cat >"$SYSCTL" <<EOF
# IP spoofing protection / rp_filter
net.ipv4.conf.all.rp_filter=1
net.ipv4.conf.default.rp_filter=1

# Redirects / source routing
net.ipv4.conf.all.accept_redirects=0
net.ipv4.conf.default.accept_redirects=0
net.ipv4.conf.all.send_redirects=0
net.ipv4.conf.default.send_redirects=0
net.ipv4.conf.all.accept_source_route=0
net.ipv4.conf.default.accept_source_route=0

# ICMP
net.ipv4.icmp_echo_ignore_broadcasts=1
net.ipv4.icmp_ignore_bogus_error_responses=1

# Log martians, syncookies
net.ipv4.conf.all.log_martians=1
net.ipv4.tcp_syncookies=1

# IPv6 (if enabled)
net.ipv6.conf.all.accept_redirects=0
net.ipv6.conf.default.accept_redirects=0
net.ipv6.conf.all.accept_ra=0
net.ipv6.conf.default.accept_ra=0
EOF
sysctl --system

if [[ "$DISABLE_IPV6" == "yes" ]]; then
  log "Disabling IPv6…"
  cat >>"$SYSCTL" <<'EOF'
net.ipv6.conf.all.disable_ipv6=1
net.ipv6.conf.default.disable_ipv6=1
EOF
  sysctl --system
fi

# ======= 12) Filesystems & mounts ===========================================
log "Hardening mounts…"
FSTAB="/etc/fstab"
edit "$FSTAB"

# secure /tmp and /dev/shm (tmpfs flags)
if [[ "$SEPARATE_TMP_TMPFS" == "yes" ]]; then
  grep -qE '^\s*tmpfs\s+/tmp' "$FSTAB" || echo "tmpfs /tmp tmpfs defaults,rw,nosuid,nodev,noexec,relatime 0 0" >> "$FSTAB"
fi
grep -qE '^\s*tmpfs\s+/dev/shm' "$FSTAB" || echo "tmpfs /dev/shm tmpfs defaults,rw,nosuid,nodev,noexec,relatime 0 0" >> "$FSTAB"

# disable uncommon filesystems
log "Blacklisting uncommon filesystems…"
edit /etc/modprobe.d/hardening.conf
for fs in cramfs freevxfs jffs2 hfs hfsplus squashfs udf vfat; do
  grep -qs "^install $fs /bin/true" /etc/modprobe.d/hardening.conf || echo "install $fs /bin/true" >> /etc/modprobe.d/hardening.conf
done
if [[ "$DISABLE_USB_STORAGE" == "yes" ]]; then
  echo "install usb-storage /bin/true" >> /etc/modprobe.d/hardening.conf
fi

# ======= 13) File permissions on sensitive files =============================
log "Securing sensitive file permissions…"
chown root:root /etc/passwd /etc/shadow /etc/group /etc/gshadow || true
chmod 0644 /etc/passwd /etc/group
chmod 0640 /etc/shadow /etc/gshadow || true
chmod -R go-rwx /etc/ssh/ssh_host_*_key || true
chmod -R go-r /etc/ssh/ssh_host_*_key.pub || true

# ======= 14) Logging (journald) =============================================
log "Enabling persistent journald…"
mkdir -p /var/log/journal
edit /etc/systemd/journald.conf
ensure_line /etc/systemd/journald.conf "Storage=persistent"
ensure_line /etc/systemd/journald.conf "Compress=yes"
ensure_line /etc/systemd/journald.conf "ForwardToSyslog=no"
systemctl restart systemd-journald

# ======= 15) Banners =========================================================
log "Setting login banners…"
echo "$BANNER_TEXT" >/etc/issue
echo "$BANNER_TEXT" >/etc/issue.net

# ======= 16) at/cron restriction ============================================
if [[ "$RESTRICT_AT_CRON" == "yes" ]]; then
  log "Restricting at/cron access…"
  rm -f /etc/cron.deny /etc/at.deny
  echo "root" >/etc/cron.allow
  echo "root" >/etc/at.allow
  chmod 0640 /etc/cron.allow /etc/at.allow
fi

# ======= 17) Sudo hardening ==================================================
log "Hardening sudo…"
apt-get install -y sudo
edit /etc/sudoers
ensure_line /etc/sudoers "Defaults use_pty"
ensure_line /etc/sudoers "Defaults logfile=\"/var/log/sudo\""
ensure_line /etc/sudoers "Defaults timestamp_timeout=5"
# Optional: require TTY (can break some automation)
# ensure_line /etc/sudoers "Defaults requiretty"

# ======= 18) Remove/disable legacy/unused services ==========================
if [[ "$PURGE_LEGACY_SERVICES" == "yes" ]]; then
  log "Removing legacy network services…"
  apt-get purge -y xinetd inetutils-inetd telnetd telnet rsh-server rsh-client rlogin \
      talkd talk tcpd tftpd tftp vsftpd nis ypserv ypbind rwho rsh-redone-server || true
fi
if [[ "$PURGE_AVAHI_CUPS_DESKTOPY" == "yes" ]]; then
  log "Removing zeroconf/printing desktop services…"
  apt-get purge -y avahi-daemon cups cups-browsed || true
fi
systemctl disable --now bluetooth || true
systemctl disable --now rpcbind || true

# ======= 19) Remove SUID/SGID from risky binaries (conservative) =============
log "Stripping SUID/SGID bits from common binaries (conservative)…"
while read -r bin; do
  [ -e "$bin" ] && chmod a-s "$bin" || true
done <<'EOF'
/usr/bin/chsh
/usr/bin/chfn
/usr/bin/umount
/usr/bin/mount
/usr/bin/newgrp
/usr/bin/gpasswd
/bin/ping
/bin/ping6
/usr/bin/traceroute6.iputils
EOF

# ======= 20) Rsyslog (optional; journald-only by default) ====================
# Uncomment to install traditional rsyslog alongside journald
# apt-get install -y rsyslog && systemctl enable --now rsyslog

# ======= 21) Rebuild initramfs for module blacklists =========================
log "Updating initramfs…"
update-initramfs -u

# ======= 22) Final ===========================================================
log "Hardening complete. Review logs at /var/log/harden.log"
echo ">> Consider rebooting to apply mount/sysctl/module changes."
